use egui::TextStyle;
use objdiff_core::diff::{ConfigEnum, Demangler};

use crate::views::appearance::Appearance;

#[derive(Default)]
pub struct DemangleViewState {
    pub text: String,
    pub demangler: Demangler,
}

pub fn demangle_window(
    ctx: &egui::Context,
    show: &mut bool,
    state: &mut DemangleViewState,
    appearance: &Appearance,
) {
    egui::Window::new("Demangle").open(show).show(ctx, |ui| {
        egui::ComboBox::from_label("Demangler")
            .selected_text(state.demangler.name().to_string())
            .show_ui(ui, |ui| {
                for demangler in Demangler::variants() {
                    if *demangler != Demangler::None {
                        let response =
                            ui.selectable_value(&mut state.demangler, *demangler, demangler.name());
                        if let Some(description) = demangler.description() {
                            response.on_hover_text(description);
                        }
                    }
                }
            });
        ui.separator();
        ui.text_edit_singleline(&mut state.text);
        ui.add_space(10.0);
        if let Some(demangled) = state.demangler.demangle(&state.text) {
            ui.scope(|ui| {
                ui.style_mut().override_text_style = Some(TextStyle::Monospace);
                ui.colored_label(appearance.replace_color, &demangled);
            });
            if ui.button("Copy").clicked() {
                ctx.copy_text(demangled);
            }
        } else {
            ui.scope(|ui| {
                ui.style_mut().override_text_style = Some(TextStyle::Monospace);
                ui.colored_label(appearance.replace_color, "[invalid]");
            });
        }
    });
}
