/** @license React v0.0.0-experimental-7b346e4e5
 * react-fetch.node.development.js
 *
 * Copyright (c) Facebook, Inc. and its affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */

'use strict';

if (process.env.NODE_ENV !== "production") {
  (function() {
'use strict';

var http = require('http');
var https = require('https');
var unstableCache = require('react/unstable-cache');

function nodeFetch(url, options, onResolve, onReject) {
  var _URL = new URL(url),
      hostname = _URL.hostname,
      pathname = _URL.pathname,
      search = _URL.search,
      port = _URL.port,
      protocol = _URL.protocol;

  var nodeOptions = {
    hostname: hostname,
    port: port,
    path: pathname + search // TODO: cherry-pick supported user-passed options.

  };
  var nodeImpl = protocol === 'https:' ? https : http;
  var request = nodeImpl.request(nodeOptions, function (response) {
    // TODO: support redirects.
    onResolve(new Response(response));
  });
  request.on('error', function (error) {
    onReject(error);
  });
  request.end();
}

var Pending = 0;
var Resolved = 1;
var Rejected = 2;
var fetchKey = {};

function readResultMap() {
  var resources = unstableCache.readCache().resources;
  var map = resources.get(fetchKey);

  if (map === undefined) {
    map = new Map();
    resources.set(fetchKey, map);
  }

  return map;
}

function readResult(result) {
  if (result.status === Resolved) {
    return result.value;
  } else {
    throw result.value;
  }
}

function Response(nativeResponse) {
  this.headers = nativeResponse.headers;
  this.ok = nativeResponse.statusCode >= 200 && nativeResponse.statusCode < 300;
  this.redirected = false; // TODO

  this.status = nativeResponse.statusCode;
  this.statusText = nativeResponse.statusMessage;
  this.type = 'basic';
  this.url = nativeResponse.url;
  this._response = nativeResponse;
  this._blob = null;
  this._json = null;
  this._text = null;
  var callbacks = [];

  function wake() {
    // This assumes they won't throw.
    while (callbacks.length > 0) {
      var cb = callbacks.pop();
      cb();
    }
  }

  var result = this._result = {
    status: Pending,
    value: {
      then: function (cb) {
        callbacks.push(cb);
      }
    }
  };
  var data = [];
  nativeResponse.on('data', function (chunk) {
    return data.push(chunk);
  });
  nativeResponse.on('end', function () {
    if (result.status === Pending) {
      var resolvedResult = result;
      resolvedResult.status = Resolved;
      resolvedResult.value = Buffer.concat(data);
      wake();
    }
  });
  nativeResponse.on('error', function (err) {
    if (result.status === Pending) {
      var rejectedResult = result;
      rejectedResult.status = Rejected;
      rejectedResult.value = err;
      wake();
    }
  });
}

Response.prototype = {
  constructor: Response,
  arrayBuffer: function () {
    var buffer = readResult(this._result);
    return buffer;
  },
  blob: function () {
    // TODO: Is this needed?
    throw new Error('Not implemented.');
  },
  json: function () {
    var buffer = readResult(this._result);
    return JSON.parse(buffer.toString());
  },
  text: function () {
    var buffer = readResult(this._result);
    return buffer.toString();
  }
};

function preloadResult(url, options) {
  var map = readResultMap();
  var entry = map.get(url);

  if (!entry) {
    if (options) {
      if (options.method || options.body || options.signal) {
        // TODO: wire up our own cancellation mechanism.
        // TODO: figure out what to do with POST.
        throw Error('Unsupported option');
      }
    }

    var callbacks = [];
    var wakeable = {
      then: function (cb) {
        callbacks.push(cb);
      }
    };

    var wake = function () {
      // This assumes they won't throw.
      while (callbacks.length > 0) {
        var cb = callbacks.pop();
        cb();
      }
    };

    var result = entry = {
      status: Pending,
      value: wakeable
    };
    nodeFetch(url, options, function (response) {
      if (result.status === Pending) {
        var resolvedResult = result;
        resolvedResult.status = Resolved;
        resolvedResult.value = response;
        wake();
      }
    }, function (err) {
      if (result.status === Pending) {
        var rejectedResult = result;
        rejectedResult.status = Rejected;
        rejectedResult.value = err;
        wake();
      }
    });
    map.set(url, entry);
  }

  return entry;
}

function preload(url, options) {
  preloadResult(url, options); // Don't return anything.
}
function fetch(url, options) {
  var result = preloadResult(url, options);
  return readResult(result);
}

exports.fetch = fetch;
exports.preload = preload;
  })();
}
