package com.example.operator.frontend

import com.example.operator.frontend.FrontendReconciler.Companion.FAILURE_THRESHOLD
import com.example.operator.frontend.FrontendReconciler.Companion.HTTP_PORT
import com.example.operator.frontend.FrontendReconciler.Companion.INITIAL_DELAY_SECONDS
import com.example.operator.frontend.FrontendReconciler.Companion.PERIOD_SECONDS
import com.example.operator.frontend.FrontendReconciler.Companion.SELECTOR
import com.example.operator.frontend.FrontendReconciler.Companion.USER_ID
import com.example.operator.frontend.FrontendReconciler.Companion.labels
import com.example.operator.frontend.FrontendReconciler.Companion.metadata
import com.example.operator.frontend.FrontendReconciler.Companion.path
import com.example.operator.frontend.FrontendReconciler.Companion.selectorLabels
import com.example.operator.frontend.crd.Frontend
import io.fabric8.kubernetes.api.model.CapabilitiesBuilder
import io.fabric8.kubernetes.api.model.ContainerBuilder
import io.fabric8.kubernetes.api.model.ContainerPortBuilder
import io.fabric8.kubernetes.api.model.EnvVar
import io.fabric8.kubernetes.api.model.EnvVarBuilder
import io.fabric8.kubernetes.api.model.HTTPGetActionBuilder
import io.fabric8.kubernetes.api.model.IntOrString
import io.fabric8.kubernetes.api.model.LabelSelectorBuilder
import io.fabric8.kubernetes.api.model.ObjectMetaBuilder
import io.fabric8.kubernetes.api.model.PodSecurityContextBuilder
import io.fabric8.kubernetes.api.model.ProbeBuilder
import io.fabric8.kubernetes.api.model.ResourceRequirementsBuilder
import io.fabric8.kubernetes.api.model.SeccompProfileBuilder
import io.fabric8.kubernetes.api.model.SecurityContextBuilder
import io.fabric8.kubernetes.api.model.apps.Deployment
import io.fabric8.kubernetes.api.model.apps.DeploymentBuilder
import io.fabric8.kubernetes.api.model.apps.DeploymentStrategyBuilder
import io.javaoperatorsdk.operator.api.reconciler.Context
import io.javaoperatorsdk.operator.processing.dependent.kubernetes.CRUDKubernetesDependentResource
import io.javaoperatorsdk.operator.processing.dependent.kubernetes.KubernetesDependent

@KubernetesDependent(labelSelector = SELECTOR)
internal class DeploymentDependentResource :
    CRUDKubernetesDependentResource<Deployment, Frontend>(Deployment::class.java) {
    @Suppress("Detekt.LongMethod", "Detekt.CyclomaticComplexMethod", "Detekt.NestedBlockDepth")
    override fun desired(primary: Frontend, context: Context<Frontend>): Deployment {
        val existing = context.getSecondaryResource(Deployment::class.java).orElse(null)
        return if (existing == null) {
            DeploymentBuilder().apply(primary).build()
        } else {
            DeploymentBuilder().apply(primary).build()
        }
    }

    private fun DeploymentBuilder.apply(primary: Frontend): DeploymentBuilder =
        withMetadata(metadata(primary))
            .withNewSpec()
            .withReplicas(1)
            .withStrategy(DeploymentStrategyBuilder().withType("Recreate").build())
            .withSelector(LabelSelectorBuilder().withMatchLabels<String, String>(selectorLabels(primary)).build())
            .withNewTemplate()
            .withMetadata(ObjectMetaBuilder().withLabels<String, String>(labels(primary)).build())
            .withNewSpec()
            .withServiceAccountName("default")
            .withSecurityContext(PodSecurityContextBuilder().withFsGroup(USER_ID).build())
            .withContainers(
                ContainerBuilder()
                    .withName("app")
                    .withImage(image(primary))
                    .withImagePullPolicy("IfNotPresent")
                    .withEnv(env(primary))
                    .withReadinessProbe(
                        ProbeBuilder()
                            .withHttpGet(
                                HTTPGetActionBuilder()
                                    .withPath(path(primary))
                                    .withPort(IntOrString("http"))
                                    .withScheme("HTTP")
                                    .build()
                            )
                            .withFailureThreshold(FAILURE_THRESHOLD)
                            .withInitialDelaySeconds(INITIAL_DELAY_SECONDS)
                            .withPeriodSeconds(PERIOD_SECONDS)
                            .withSuccessThreshold(1)
                            .withTimeoutSeconds(1)
                            .build()
                    )
                    .withResources(ResourceRequirementsBuilder().build())
                    .withSecurityContext(
                        SecurityContextBuilder()
                            .withAllowPrivilegeEscalation(false)
                            .withCapabilities(CapabilitiesBuilder().withDrop("ALL").build())
                            .withReadOnlyRootFilesystem(false)
                            .withRunAsGroup(USER_ID)
                            .withRunAsUser(USER_ID)
                            .withRunAsNonRoot(true)
                            .withSeccompProfile(SeccompProfileBuilder().withType("RuntimeDefault").build())
                            .build()
                    )
                    .withPorts(
                        ContainerPortBuilder()
                            .withName("http")
                            .withContainerPort(HTTP_PORT)
                            .withProtocol("TCP")
                            .build(),
                    )
                    .build()
            )
            .endSpec()
            .endTemplate()
            .endSpec()

    private fun env(primary: Frontend): List<EnvVar> =
        primary.spec.env.plus(EnvVarBuilder().withName("BASE_PATH").withValue(path(primary)).build())

    private fun image(primary: Frontend): String =
        with(primary.spec) {
            val repository = "$project/$name"
            val tag = version
            "${image.registry}/$repository:$tag"
        }
}
