package com.example.operator.frontend

import com.example.operator.EnvironmentVariables.domain
import com.example.operator.EnvironmentVariables.tlsSecretName
import com.example.operator.frontend.StatusTypes.READY
import com.example.operator.frontend.crd.Frontend
import com.example.operator.frontend.crd.FrontendStatus
import io.fabric8.kubernetes.api.model.Condition
import io.fabric8.kubernetes.api.model.ConditionBuilder
import io.fabric8.kubernetes.api.model.ObjectMeta
import io.fabric8.kubernetes.api.model.apps.Deployment
import io.fabric8.kubernetes.client.readiness.Readiness.isDeploymentReady
import io.github.oshai.kotlinlogging.KotlinLogging
import io.javaoperatorsdk.operator.api.reconciler.Context
import io.javaoperatorsdk.operator.api.reconciler.ControllerConfiguration
import io.javaoperatorsdk.operator.api.reconciler.ErrorStatusHandler
import io.javaoperatorsdk.operator.api.reconciler.ErrorStatusUpdateControl
import io.javaoperatorsdk.operator.api.reconciler.Reconciler
import io.javaoperatorsdk.operator.api.reconciler.UpdateControl
import io.javaoperatorsdk.operator.api.reconciler.dependent.Dependent

private val logger = KotlinLogging.logger {}

@ControllerConfiguration(dependents = [Dependent(type = DeploymentDependentResource::class)])
class FrontendReconciler : Reconciler<Frontend>, ErrorStatusHandler<Frontend> {
    override fun reconcile(
        resource: Frontend,
        context: Context<Frontend>
    ): UpdateControl<Frontend> {
        logger.debug { "Reconciling ${resource.metadata.name}" }

        val ready = isDeploymentReady(context.getSecondaryResource(Deployment::class.java).orElseThrow())
        val url = ingressUrl(resource)

        val condition =
            if (ready) {
                ConditionBuilder().withType(READY).withStatus("True").build()
            } else {
                ConditionBuilder()
                    .withType(READY)
                    .withStatus("Unknown")
                    .withReason("Creating")
                    .withMessage("Resources are creating")
                    .build()
            }

        val changed = updateStatus(resource, url, condition)
        return if (changed) {
            UpdateControl.patchStatus(resource)
        } else {
            UpdateControl.noUpdate()
        }
    }

    override fun updateErrorStatus(
        resource: Frontend,
        context: Context<Frontend>,
        error: Exception
    ): ErrorStatusUpdateControl<Frontend> {
        val url = ingressUrl(resource)

        val condition =
            ConditionBuilder()
                .withType(READY)
                .withStatus("False")
                .withReason("DeploymentFailed")
                .withMessage(error.localizedMessage)
                .build()

        val changed = updateStatus(resource, url, condition)
        return if (changed) {
            ErrorStatusUpdateControl.patchStatus(resource)
        } else {
            ErrorStatusUpdateControl.noStatusUpdate()
        }
    }

    private fun updateStatus(resource: Frontend, url: String?, condition: Condition): Boolean {
        if (resource.status == null) {
            resource.status = FrontendStatus()
        }
        if (resource.status.url != url) {
            resource.status.url = url
        }
        return resource.status.add(condition)
    }

    private fun ingressUrl(resource: Frontend): String {
        val protocol = if (tlsSecretName != null) "https" else "http"
        return "${protocol}://${host(resource)}${path(resource)}"
    }

    companion object {
        const val SELECTOR: String = "example.com/managed"
        const val USER_ID = 1000L
        const val HTTP_PORT = 8080
        const val FAILURE_THRESHOLD = 6
        const val INITIAL_DELAY_SECONDS = 1
        const val PERIOD_SECONDS = 5

        fun metadata(resource: Frontend): ObjectMeta =
            ObjectMeta().apply {
                name = resource.metadata.name
                namespace = resource.metadata.namespace
                labels = labels((resource))
            }

        fun host(primary: Frontend): String {
            val suffix = primary.spec.ingress?.domain ?: domain
            return "${primary.spec.environment}-${primary.spec.project}${if (suffix != null) ".${suffix}" else ""}"
        }

        fun path(primary: Frontend): String = "/apps/${primary.spec.name}"

        fun selectorLabels(resource: Frontend): Map<String, String> =
            mapOf(
                "app.kubernetes.io/instance" to resource.metadata.name,
                "app.kubernetes.io/name" to resource.spec.name,
            )

        fun labels(resource: Frontend): Map<String, String> = selectorLabels(resource).plus(additionalLabels(resource))

        private fun additionalLabels(resource: Frontend): Map<String, String> =
            mapOf(
                "app.kubernetes.io/version" to resource.spec.version,
                SELECTOR to "true",
            )
    }
}
